from pyparsing import Literal, printables, Word

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, Detector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.docker.docker_detectors import DockerImageBasedComponent
from codeable_detectors.evidences import FailedEvidence, Evidence, ComponentEvidence, LinkEvidence, Evidences, \
    ServiceEvidence
from codeable_detectors.pyparsing_patterns import curly_braces_block, http_url_start
from codeable_detectors.utils import get_required_keyword_arg


class NginxLocations(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["conf", "conf.template"]

    def detect_in_context(self, ctx, **kwargs):
        server_block_pattern = Literal("server") + curly_braces_block
        location_pattern = Literal("location") + Word(printables, excludeChars="}{") + curly_braces_block

        matches = []
        server_block_matches = ctx.matches_pattern(server_block_pattern)
        for serverBlockMatch in server_block_matches:
            matches.extend(DetectorContext(serverBlockMatch).matches_pattern(location_pattern))
        if matches:
            return Evidence(matches)
        return FailedEvidence("nginx location block not found")


class NginxLocationsToRESTAPIClient(NginxLocations):
    def detect_in_context(self, ctx, **kwargs):
        # TODO: here caching the result on project might be an option in order not to run NginxLocations 3 times
        location_evidence = super().detect_in_context(ctx)
        if location_evidence.has_succeeded():
            return ComponentEvidence(location_evidence.matches).set_properties(
                detector_name="REST API Client", detector_component_types="client",
                detector_link_types="restfulHTTP", detector_technology_types="nginx", kwargs=kwargs)
        return FailedEvidence("could not find nginx locations in file '" + ctx.file_name + "'")


class NginxLocationsToWebUIClient(NginxLocations):
    def detect_in_context(self, ctx, **kwargs):
        # TODO: here caching the result on project might be an option in order not to run NginxLocations 3 times
        location_evidence = super().detect_in_context(ctx)

        root_pattern = Literal("root") + Word(printables, excludeChars=";") + Literal(";")
        matches = []

        for location_match in location_evidence.matches:
            root_matches = DetectorContext(location_match).matches_pattern(root_pattern)
            if len(root_matches) > 0:
                matches.append(location_match)
        if matches:
            return ComponentEvidence(matches).set_properties(detector_name="Web UI Client",
                                                             detector_component_types="webUI",
                                                             detector_link_types="http",
                                                             detector_technology_types="nginx", kwargs=kwargs)
        return FailedEvidence("could not find nginx locations that contain a web root in file '" +
                              ctx.file_name + "'")


class NginxLocationToServiceLink(NginxLocations):
    def detect_in_context(self, ctx, **kwargs):
        target = get_required_keyword_arg('target', kwargs)

        # use identifier to guess whether this is a location pointing to this link
        target_identifier = target.identifier

        location_evidence = super().detect_in_context(ctx)

        location_start_pattern = Literal("location") + Word(printables, excludeChars="}{")
        matches = []

        location_matches = False
        for location_match in location_evidence.matches:
            location_start_matches = DetectorContext(location_match).matches_pattern(location_start_pattern)
            for location_start_match in location_start_matches:
                url_location = location_start_match.text[8:].strip()
                for alias in target.aliases:
                    if alias.lower() in url_location.lower():
                        matches.append(location_start_match)
                        location_matches = True
        # if the location does not yet match, match the host pattern
        if not location_matches:
            for alias in target.aliases:
                proxy_pass_matches = detect_proxy_pass_in_nginx_location(location_evidence.matches, alias)
                matches.extend(proxy_pass_matches)

        if not matches:
            return FailedEvidence("could not find location that matches '" + target_identifier + "'")

        return LinkEvidence(matches).set_properties(detector_link_types="restfulHTTP",
                                                    detector_technology_types="nginx", kwargs=kwargs)


def detect_proxy_pass_in_nginx_location(location_matches, url_part=None):
    pattern = Literal("proxy_pass") + http_url_start + Word(printables, excludeChars=";") + Literal(";")
    matches = []

    for location_match in location_matches:
        location_ctx = DetectorContext(location_match)
        proxy_pass_matches = location_ctx.matches_pattern(pattern)
        for match in proxy_pass_matches:
            url = match.text[10:-1]
            if url_part is not None:
                if url_part.lower() in url.lower():
                    match.update_keyword_args(url=url)
                    matches.append(match)
            else:
                match.update_keyword_args(url=url)
                matches.append(match)
    return matches


class NginxProxyPass(NginxLocations):
    def detect_in_context(self, ctx, **kwargs):
        location_evidence = super().detect_in_context(ctx)
        proxy_pass_matches = detect_proxy_pass_in_nginx_location(location_evidence.matches)
        if proxy_pass_matches:
            return Evidence(proxy_pass_matches)
        return FailedEvidence("nginx location blocks contain no proxy_pass fields")


class NginxAPIGateway(Detector):
    def __init__(self):
        super().__init__()
        self.nginx_present_in_docker_detector = DockerImageBasedComponent("nginx", is_exposed=True)
        self.proxy_pass_detector = NginxProxyPass()

    def detect(self, directory, **kwargs):
        # here an alternative detection for nginx being present but not 
        # dockerized should be added.
        nginx_present_evidences = self.nginx_present_in_docker_detector.detect(directory, **kwargs)
        if nginx_present_evidences.have_failed():
            return nginx_present_evidences

        proxy_pass_evidences = self.proxy_pass_detector.detect(directory, **kwargs)
        if proxy_pass_evidences.have_failed():
            return proxy_pass_evidences

        return Evidences(ServiceEvidence(nginx_present_evidences.get_all_matches() +
                                         proxy_pass_evidences.get_all_matches()).set_properties(
            detector_name="NGINX API Gateway",
            detector_component_types="facade", detector_link_types="http",
            detector_technology_types="nginx", kwargs=kwargs))
